<?php
class AuditTrailSettings {
    public static function init() {
        add_action('admin_menu', [__CLASS__, 'add_settings_page']);
        add_action('admin_post_audit_trail_save_settings', [__CLASS__, 'save_settings']);
    }

    public static function add_settings_page() {
        add_menu_page(
            'Audit Trail Settings',
            'Audit Settings',
            'manage_options',
            'audit-trail-settings',
            [__CLASS__, 'render_settings_page'],
            'dashicons-shield-alt',
            80
        );
    }

    public static function render_settings_page() {
        if (!current_user_can('manage_options')) return;

        $allowed_view_roles   = get_option('audit_trail_allowed_roles', []);
        $allowed_delete_roles = get_option('audit_trail_delete_own_logs_roles', []);
        $retention_period     = get_option('audit_trail_log_retention_months', 1);
        $roles                = wp_roles()->roles;
        $users                = get_users(['fields' => ['ID', 'user_login']]);
        ?>
        <div class="wrap">
            <h1>Audit Trail Settings</h1>
            <form method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>">
                <?php wp_nonce_field('audit_trail_save_settings'); ?>
                <input type="hidden" name="action" value="audit_trail_save_settings">

                <!-- 1. Log off any user -->
                <h2>1. Log off any user</h2>
                <div style="display: inline-flex; align-items: center; gap: 10px; margin-bottom: 10px;">
                    <button type="submit" name="logout_user_btn" class="button">Log Out User</button>
                    <select name="logout_user_id">
                        <option value="">Select a user</option>
                        <?php foreach ($users as $user): ?>
                            <option value="<?php echo esc_attr($user->ID); ?>"><?php echo esc_html($user->user_login); ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>

                <hr>

                <!-- 2. Who can view audit logs -->
                <h2>2. Limit who can view the logs</h2>
                <?php foreach ($roles as $role_key => $role): ?>
                    <label>
                        <input type="checkbox" name="allowed_roles[]" value="<?php echo esc_attr($role_key); ?>" <?php checked(in_array($role_key, $allowed_view_roles)); ?>>
                        <?php echo esc_html($role['name']); ?>
                    </label><br>
                <?php endforeach; ?>

                <hr>

                <h2>3. Who can access audit trail?</h2>
                <p>Same as above. Only selected roles will be able to access the audit logs.</p>

                <hr>

                <!-- 4. Who can delete their own logs -->
                <h2>4. Who can delete their own logs?</h2>
                <?php foreach ($roles as $role_key => $role): ?>
                    <label>
                        <input type="checkbox" name="delete_own_logs_roles[]" value="<?php echo esc_attr($role_key); ?>" <?php checked(in_array($role_key, $allowed_delete_roles)); ?>>
                        <?php echo esc_html($role['name']); ?>
                    </label><br>
                <?php endforeach; ?>

                <hr>

                <!-- 5. Retention setting -->
                <h2>5. Keep logs for (Months)</h2>
                <input type="number" name="log_retention" value="<?php echo esc_attr($retention_period); ?>" min="1" max="12">
                <p class="description">Default is 1 month</p>

                <hr>

                <!-- 6. Delete All Logs -->
                <h2>6. Delete Logs</h2>
                <p>Click the button below to delete all logs. Only accessible by administrators.</p>
                <button type="submit" name="delete_all_logs" class="button button-danger" onclick="return confirm('Are you sure you want to delete all logs? This action cannot be undone.');">Delete All Logs</button>

                <?php submit_button('Save Settings'); ?>
            </form>
        </div>
        <?php
    }

    public static function save_settings() {
        if (!current_user_can('manage_options') || !check_admin_referer('audit_trail_save_settings')) {
            wp_die('Unauthorized request');
        }

        // Save view roles
        $allowed_roles = isset($_POST['allowed_roles']) ? array_map('sanitize_text_field', (array) $_POST['allowed_roles']) : [];
        update_option('audit_trail_allowed_roles', $allowed_roles);

        // Save delete-own roles
        $delete_roles = isset($_POST['delete_own_logs_roles']) ? array_map('sanitize_text_field', (array) $_POST['delete_own_logs_roles']) : [];
        update_option('audit_trail_delete_own_logs_roles', $delete_roles);

        // Save retention period
        $retention = isset($_POST['log_retention']) ? max(1, intval($_POST['log_retention'])) : 1;
        update_option('audit_trail_log_retention_months', $retention);

        // Handle logout user
        if (!empty($_POST['logout_user_id']) && current_user_can('administrator')) {
            self::logout_selected_user(intval($_POST['logout_user_id']));
        }

        // Handle delete logs
        if (isset($_POST['delete_all_logs']) && current_user_can('administrator')) {
            self::delete_all_logs();
        }

        wp_redirect(admin_url('admin.php?page=audit-trail-settings&updated=true'));
        exit;
    }

    private static function delete_all_logs() {
        global $wpdb;
        $wpdb->query("DELETE FROM {$wpdb->prefix}audit_trail_login_history WHERE 1=1");
        $wpdb->query("DELETE FROM {$wpdb->prefix}audit_trail_user_activity WHERE 1=1");
    }

    private static function logout_selected_user($user_id) {
        if (!$user_id) return;

        if (class_exists('WP_Session_Tokens')) {
            $manager = WP_Session_Tokens::get_instance($user_id);
            $manager->destroy_all();
        }
    }

    public static function can_view_logs() {
        $allowed_roles = get_option('audit_trail_allowed_roles', []);
        $user = wp_get_current_user();
        foreach ($user->roles as $role) {
            if (in_array($role, $allowed_roles)) return true;
        }
        return false;
    }

    public static function can_delete_own_logs() {
        $allowed_roles = get_option('audit_trail_delete_own_logs_roles', []);
        $user = wp_get_current_user();
        foreach ($user->roles as $role) {
            if (in_array($role, $allowed_roles)) return true;
        }
        return false;
    }

    public static function get_log_retention_days() {
        $months = get_option('audit_trail_log_retention_months', 1);
        return $months * 30;
    }
}

AuditTrailSettings::init();
